/*
 * Decompiled with CFR 0.152.
 */
package umcg.genetica.io.binInteraction;

import com.google.common.io.CountingInputStream;
import gnu.trove.map.hash.TObjectIntHashMap;
import java.io.BufferedInputStream;
import java.io.Closeable;
import java.io.DataInputStream;
import java.io.EOFException;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import org.molgenis.genotype.Allele;
import umcg.genetica.collections.ChrPosMap;
import umcg.genetica.io.binInteraction.BinaryInteractionCohort;
import umcg.genetica.io.binInteraction.BinaryInteractionCovariateIterator;
import umcg.genetica.io.binInteraction.BinaryInteractionFileConstructorBuilder;
import umcg.genetica.io.binInteraction.BinaryInteractionFileException;
import umcg.genetica.io.binInteraction.BinaryInteractionQtlZscores;
import umcg.genetica.io.binInteraction.BinaryInteractionQueryResult;
import umcg.genetica.io.binInteraction.BinaryInteractionZscores;
import umcg.genetica.io.binInteraction.gene.BinaryInteractionGene;
import umcg.genetica.io.binInteraction.gene.BinaryInteractionGeneStatic;
import umcg.genetica.io.binInteraction.variant.BinaryInteractionVariant;
import umcg.genetica.io.binInteraction.variant.BinaryInteractionVariantStatic;

public class BinaryInteractionFile
implements Closeable {
    protected static final byte MAGIC_1 = 81;
    protected static final byte MAGIC_2 = 73;
    private static final long POINTER_TO_CLOSED_BOOLEAN = 4L;
    private static final byte MAJOR_VERSION = 1;
    private static final byte MINOR_VERSION = 0;
    private static final int NO_ENTRY_INT_MAP = -1;
    private static final SimpleDateFormat DEFAULT_DATE_FORMAT = new SimpleDateFormat("dd-MM-yyyy HH:mm:ss");
    private static final int BUFFER_SIZE = 8192;
    private final File interactionFile;
    private boolean readOnly;
    private final BinaryInteractionCohort[] cohorts;
    private final BinaryInteractionGene[] genes;
    private final BinaryInteractionVariant[] variants;
    private ChrPosMap<BinaryInteractionVariant> variantsByPos = null;
    protected final String[] covariates;
    private final int[][] covariatesTested;
    private final long timeStamp;
    private final boolean allCovariants;
    private final boolean metaAnalysis;
    private final boolean normalQtlStored;
    private final boolean flippedZscoreStored;
    private final String fileDescription;
    private final long interactions;
    private final long startQtlBlock;
    private final long startInteractionBlock;
    private final long sizeQtlBlock;
    protected final long sizeInteractionBlock;
    private final int[] cummulativeGeneCountUpToVariant;
    private final long[] cummalitiveInteractionCountUptoVariantGene;
    private final TObjectIntHashMap<String> variantMap;
    private final TObjectIntHashMap<String> genesMap;
    private final TObjectIntHashMap<String> covariatesMap;
    private RandomAccessFile randomAccess;
    private FileChannel channel;
    private final ByteBuffer qtlBuffer = ByteBuffer.allocate(8192);
    private final ByteBuffer interactionBuffer = ByteBuffer.allocate(8192);
    private boolean qtlBufferWriting = false;
    private boolean interactionBufferWriting = false;
    private long qtlBufferStart = Long.MIN_VALUE;
    private long interactionBufferStart = Long.MIN_VALUE;
    private long qtlZscoresSet = 0L;
    private long interactionZscoresSet = 0L;
    private long qtlZscoresRead = 0L;
    private long interactionZscoresRead = 0L;
    private long interactionWriteBufferFlushed = 0L;
    private long qtlWriteBufferFlushed = 0L;
    private long interactionReadBufferLoaded = 0L;
    private long qtlReadBufferLoaded = 0L;

    protected BinaryInteractionFile(File interactionFile, boolean readOnly, BinaryInteractionCohort[] cohorts, BinaryInteractionGene[] genes, BinaryInteractionVariant[] variants, String[] covariates, int[][] covariatesTested, long timeStamp, boolean allCovariants, boolean metaAnalysis, boolean normalQtlStored, boolean flippedZscoreStored, String fileDescription, long interactions, long startQtlBlock, long startInteractionBlock) throws BinaryInteractionFileException, FileNotFoundException, IOException {
        int i;
        this.interactionFile = interactionFile;
        this.readOnly = readOnly;
        this.cohorts = cohorts;
        this.genes = genes;
        this.variants = variants;
        this.covariates = covariates;
        this.covariatesTested = covariatesTested;
        this.timeStamp = timeStamp;
        this.allCovariants = allCovariants;
        this.metaAnalysis = metaAnalysis;
        this.normalQtlStored = normalQtlStored;
        this.flippedZscoreStored = flippedZscoreStored;
        this.fileDescription = fileDescription;
        this.interactions = interactions;
        this.startQtlBlock = startQtlBlock;
        this.startInteractionBlock = startInteractionBlock;
        this.sizeQtlBlock = BinaryInteractionFile.calculateSizeNormalQtlBlock(cohorts.length, metaAnalysis);
        this.sizeInteractionBlock = BinaryInteractionFile.calculateSizeInteractionResultBlock(cohorts.length, flippedZscoreStored, metaAnalysis);
        this.cummulativeGeneCountUpToVariant = new int[variants.length + 1];
        for (i = 0; i < variants.length; ++i) {
            this.cummulativeGeneCountUpToVariant[i + 1] = this.cummulativeGeneCountUpToVariant[i] + variants[i].getGeneCount();
        }
        this.cummalitiveInteractionCountUptoVariantGene = new long[this.cummulativeGeneCountUpToVariant[variants.length] + 1];
        i = 1;
        for (int v = 0; v < variants.length; ++v) {
            int variantGeneCount = variants[v].getGeneCount();
            for (int g = 0; g < variantGeneCount; ++g) {
                this.cummalitiveInteractionCountUptoVariantGene[i] = this.cummalitiveInteractionCountUptoVariantGene[i - 1] + (long)(allCovariants ? covariates.length : this.covariatesTested[i - 1].length);
                ++i;
            }
        }
        if (this.cummalitiveInteractionCountUptoVariantGene[this.cummulativeGeneCountUpToVariant[variants.length]] != interactions) {
            throw new BinaryInteractionFileException("Something went wrong");
        }
        this.variantMap = new TObjectIntHashMap(variants.length, 0.75f, -1);
        this.genesMap = new TObjectIntHashMap(genes.length, 0.75f, -1);
        this.covariatesMap = new TObjectIntHashMap(covariates.length, 0.75f, -1);
        for (i = 0; i < variants.length; ++i) {
            if (this.variantMap.put((Object)variants[i].getName(), i) == -1) continue;
            throw new BinaryInteractionFileException("Cannot store the same variant twice (" + variants[i].getName() + ")");
        }
        for (i = 0; i < genes.length; ++i) {
            if (this.genesMap.put((Object)genes[i].getName(), i) == -1) continue;
            throw new BinaryInteractionFileException("Cannot store the same gene twice (" + genes[i].getName() + ")");
        }
        for (i = 0; i < covariates.length; ++i) {
            if (this.covariatesMap.put((Object)covariates[i], i) == -1) continue;
            throw new BinaryInteractionFileException("Cannot store the same covariate twice (" + covariates[i] + ")");
        }
        if (this.sizeQtlBlock >= 8192L) {
            throw new BinaryInteractionFileException("QTL block size larger than buffer");
        }
        if (this.sizeInteractionBlock >= 8192L) {
            throw new BinaryInteractionFileException("Interaction block size larger than buffer");
        }
        this.open();
    }

    public static BinaryInteractionFile load(File interactionFile) throws FileNotFoundException, IOException, BinaryInteractionFileException {
        return BinaryInteractionFile.load(interactionFile, true);
    }

    public static BinaryInteractionFile load(File interactionFile, boolean readOnly) throws FileNotFoundException, IOException, BinaryInteractionFileException {
        BinaryInteractionFileConstructorBuilder builder = new BinaryInteractionFileConstructorBuilder();
        builder.setInteractionFile(interactionFile);
        builder.setReadOnly(readOnly);
        CountingInputStream inputStreamCounted = new CountingInputStream((InputStream)new BufferedInputStream(new FileInputStream(interactionFile)));
        DataInputStream inputStream = new DataInputStream((InputStream)inputStreamCounted);
        try {
            long sizeNormalQtlSection;
            long startNormalQtlSection;
            if (inputStream.readByte() != 81 || inputStream.readByte() != 73) {
                throw new BinaryInteractionFileException("No a valid interaction file");
            }
            if (inputStream.readByte() != 1 || inputStream.readByte() != 0) {
                throw new BinaryInteractionFileException("Interaction file version not supported");
            }
            if (!inputStream.readBoolean()) {
                throw new BinaryInteractionFileException("Interaction file not properly closed and might be corrupt");
            }
            boolean allCovariants = inputStream.readBoolean();
            builder.setAllCovariants(allCovariants);
            boolean metaAnalysis = inputStream.readBoolean();
            builder.setMetaAnalysis(metaAnalysis);
            boolean normalQtlStored = inputStream.readBoolean();
            builder.setNormalQtlStored(normalQtlStored);
            boolean flippedZscoreStored = inputStream.readBoolean();
            builder.setFlippedZscoreStored(flippedZscoreStored);
            inputStream.skip(4L);
            builder.setTimeStamp(inputStream.readLong());
            builder.setFileDescription(BinaryInteractionFile.readString(inputStream));
            int chortsCount = inputStream.readInt();
            int chrsCount = inputStream.readInt();
            int allelesCount = inputStream.readInt();
            int genesCount = inputStream.readInt();
            int variantCount = inputStream.readInt();
            int covariatesCount = inputStream.readInt();
            long totalInteractions = inputStream.readLong();
            builder.setInteractions(totalInteractions);
            builder.setCohorts(BinaryInteractionFile.readCohorts(inputStream, chortsCount));
            String[] chrDictionary = BinaryInteractionFile.readStringArray(inputStream, chrsCount, "Chromosomes");
            Allele[] alleleDictionary = BinaryInteractionFile.readAlleles(inputStream, allelesCount);
            BinaryInteractionVariant[] variants = BinaryInteractionFile.readVariants(inputStream, variantCount, chrDictionary, alleleDictionary);
            builder.setVariants(variants);
            builder.setGenes(BinaryInteractionFile.readGenes(inputStream, genesCount, chrDictionary));
            builder.setCovariates(BinaryInteractionFile.readStringArray(inputStream, covariatesCount, "Chromosomes"));
            int totalVariantGeneCombinations = BinaryInteractionFile.getTotalVariantGeneCombinations(variants);
            if (!allCovariants) {
                builder.setCovariatesTested(BinaryInteractionFile.readCovariatesData(inputStream, totalVariantGeneCombinations, totalInteractions));
            }
            long startData = inputStreamCounted.getCount();
            if (normalQtlStored) {
                startNormalQtlSection = startData;
                long sizeQtlBlock = BinaryInteractionFile.calculateSizeNormalQtlBlock(chortsCount, metaAnalysis);
                sizeNormalQtlSection = sizeQtlBlock * (long)totalVariantGeneCombinations;
            } else {
                sizeNormalQtlSection = 0L;
                startNormalQtlSection = -1L;
            }
            builder.setStartQtlBlock(startNormalQtlSection);
            long startInteractionSection = startData + sizeNormalQtlSection;
            builder.setStartInteractionBlock(startInteractionSection);
            long sizeInteractionSection = BinaryInteractionFile.calculateSizeInteractionResultBlock(chortsCount, flippedZscoreStored, metaAnalysis) * totalInteractions;
            if (startData + sizeNormalQtlSection + sizeInteractionSection != interactionFile.length()) {
                throw new BinaryInteractionFileException("Incorrect file size. Expected: " + (startData + sizeNormalQtlSection + sizeInteractionSection) + " found: " + interactionFile.length() + " diff: " + (startData + sizeNormalQtlSection + sizeInteractionSection - interactionFile.length()));
            }
            inputStream.close();
            inputStreamCounted.close();
            return builder.createBinaryInteractionFile();
        }
        catch (EOFException ex) {
            throw new BinaryInteractionFileException("Error parsing header, unexpected EOF", ex);
        }
    }

    private static BinaryInteractionCohort[] readCohorts(DataInputStream inputStream, int cohortsCount) throws EOFException, IOException, BinaryInteractionFileException {
        BinaryInteractionCohort[] cohorts = new BinaryInteractionCohort[cohortsCount];
        HashSet<String> cohortNames = new HashSet<String>(cohortsCount);
        for (int i = 0; i < cohortsCount; ++i) {
            String name = BinaryInteractionFile.readString(inputStream);
            if (!cohortNames.add(name)) {
                throw new BinaryInteractionFileException("Cohort names must be unique. " + name + " has been found multiple times");
            }
            cohorts[i] = new BinaryInteractionCohort(name, inputStream.readInt());
        }
        return cohorts;
    }

    private static String[] readStringArray(DataInputStream inputStream, int size, String name) throws BinaryInteractionFileException, EOFException, IOException {
        String[] array = new String[size];
        HashSet<String> names = new HashSet<String>(size);
        for (int i = 0; i < size; ++i) {
            array[i] = BinaryInteractionFile.readString(inputStream);
            if (names.add(array[i])) continue;
            throw new BinaryInteractionFileException(name + " entries must be unique. " + array[i] + " has been found multiple times");
        }
        return array;
    }

    private static Allele[] readAlleles(DataInputStream inputStream, int size) throws BinaryInteractionFileException, EOFException, IOException {
        Allele[] alleles = new Allele[size];
        HashSet<String> allelesCheck = new HashSet<String>(size);
        for (int i = 0; i < size; ++i) {
            String alleleString = BinaryInteractionFile.readString(inputStream);
            alleles[i] = Allele.create((String)alleleString);
            if (allelesCheck.add(alleleString)) continue;
            throw new BinaryInteractionFileException("Allele entries must be unique. " + alleleString + " has been found multiple times");
        }
        return alleles;
    }

    private static BinaryInteractionVariant[] readVariants(DataInputStream inputStream, int variantCount, String[] chrDictionary, Allele[] alleleDictionary) throws EOFException, IOException, BinaryInteractionFileException {
        BinaryInteractionVariant[] variants = new BinaryInteractionVariant[variantCount];
        for (int i = 0; i < variantCount; ++i) {
            String name = BinaryInteractionFile.readString(inputStream);
            String chr = chrDictionary[inputStream.readInt()];
            int pos = inputStream.readInt();
            Allele refAllele = alleleDictionary[inputStream.readInt()];
            Allele altAllele = alleleDictionary[inputStream.readInt()];
            int geneCount = inputStream.readInt();
            int[] genes = BinaryInteractionFile.readIntArray(inputStream, geneCount);
            variants[i] = new BinaryInteractionVariantStatic(name, chr, pos, refAllele, altAllele, genes);
        }
        return variants;
    }

    private static BinaryInteractionGene[] readGenes(DataInputStream inputStream, int geneCount, String[] chrDictionary) throws EOFException, IOException {
        BinaryInteractionGene[] genes = new BinaryInteractionGene[geneCount];
        for (int i = 0; i < geneCount; ++i) {
            String name = BinaryInteractionFile.readString(inputStream);
            String chr = chrDictionary[inputStream.readInt()];
            int start = inputStream.readInt();
            int end = inputStream.readInt();
            int variantCount = inputStream.readInt();
            int[] variants = BinaryInteractionFile.readIntArray(inputStream, variantCount);
            genes[i] = new BinaryInteractionGeneStatic(name, chr, start, end, variants);
        }
        return genes;
    }

    private static int[][] readCovariatesData(DataInputStream inputStream, int totalSnpGeneCombinations, long totalInteractions) throws BinaryInteractionFileException, EOFException, IOException {
        int[][] testedCovariates = new int[totalSnpGeneCombinations][];
        long interactionSumCovariates = 0L;
        for (int i = 0; i < totalSnpGeneCombinations; ++i) {
            int covariatesCount = inputStream.readInt();
            testedCovariates[i] = BinaryInteractionFile.readIntArray(inputStream, covariatesCount);
            interactionSumCovariates += (long)testedCovariates[i].length;
        }
        if (interactionSumCovariates != totalInteractions) {
            throw new BinaryInteractionFileException("Interactions in header differs from interactions in covariate block. Header: " + totalInteractions + " covariate sum: " + interactionSumCovariates);
        }
        return testedCovariates;
    }

    private static String readString(DataInputStream inputStream) throws EOFException, IOException {
        int lenght = inputStream.readInt();
        char[] chars = new char[lenght];
        for (int i = 0; i < lenght; ++i) {
            chars[i] = inputStream.readChar();
        }
        return new String(chars);
    }

    private static int[] readIntArray(DataInputStream inputStream, int arraySize) throws EOFException, IOException {
        int[] array = new int[arraySize];
        for (int i = 0; i < arraySize; ++i) {
            array[i] = inputStream.readInt();
        }
        return array;
    }

    protected static long calculateSizeNormalQtlBlock(int cohorts, boolean metaAnalysis) {
        long size = cohorts * 12;
        if (metaAnalysis) {
            size += 8L;
        }
        return size;
    }

    protected static long calculateSizeInteractionResultBlock(int cohorts, boolean flippedZscoreStored, boolean metaAnalysis) {
        long size = cohorts * 36;
        if (flippedZscoreStored) {
            size += (long)(cohorts * 8);
            if (metaAnalysis) {
                size += 8L;
            }
        }
        if (metaAnalysis) {
            size += 24L;
        }
        return size;
    }

    private static int getTotalVariantGeneCombinations(BinaryInteractionVariant[] variants) throws BinaryInteractionFileException {
        long totalSnpGeneCombinationsLong = 0L;
        for (BinaryInteractionVariant variant : variants) {
            totalSnpGeneCombinationsLong += (long)variant.getGeneCount();
        }
        if (totalSnpGeneCombinationsLong > Integer.MAX_VALUE) {
            throw new BinaryInteractionFileException("Cannot handle variant-gene combinations bigger than 2^31-1");
        }
        return (int)totalSnpGeneCombinationsLong;
    }

    public File getInteractionFile() {
        return this.interactionFile;
    }

    public boolean isReadOnly() {
        return this.readOnly;
    }

    public long getCreationDataEpoch() {
        return this.timeStamp;
    }

    public String getCreationDataTimeString() {
        return DEFAULT_DATE_FORMAT.format(new Date(this.timeStamp * 1000L));
    }

    public boolean isMetaAnalysis() {
        return this.metaAnalysis;
    }

    public boolean isNormalQtlStored() {
        return this.normalQtlStored;
    }

    public boolean isFlippedZscoreStored() {
        return this.flippedZscoreStored;
    }

    public String getFileDescription() {
        return this.fileDescription;
    }

    public boolean areAllCovariatesTestedForAllVariantGenes() {
        return this.allCovariants;
    }

    public long getTotalNumberInteractions() {
        return this.interactions;
    }

    public List<BinaryInteractionCohort> getCohorts() {
        return Collections.unmodifiableList(Arrays.asList(this.cohorts));
    }

    public List<BinaryInteractionGene> getGenes() {
        return Collections.unmodifiableList(Arrays.asList(this.genes));
    }

    public BinaryInteractionGene getGene(String name) throws BinaryInteractionFileException {
        int index = this.genesMap.get((Object)name);
        if (index == -1) {
            throw new BinaryInteractionFileException("Gene not found: " + name);
        }
        return this.genes[index];
    }

    public BinaryInteractionGene getGene(int pointer) throws BinaryInteractionFileException {
        return this.genes[pointer];
    }

    public List<BinaryInteractionVariant> getVariants() {
        return Collections.unmodifiableList(Arrays.asList(this.variants));
    }

    public BinaryInteractionVariant getVariant(String name) throws BinaryInteractionFileException {
        int index = this.variantMap.get((Object)name);
        if (index == -1) {
            throw new BinaryInteractionFileException("Variant not found: " + name);
        }
        return this.variants[index];
    }

    public BinaryInteractionVariant getVariant(int pointer) throws BinaryInteractionFileException {
        return this.variants[pointer];
    }

    public List<String> getCovariates() {
        return Collections.unmodifiableList(Arrays.asList(this.covariates));
    }

    public void finalizeWriting() throws IOException, BinaryInteractionFileException {
        if (!this.readOnly) {
            if (this.qtlBufferWriting) {
                this.writeQtlBuffer();
            }
            if (this.interactionBufferWriting) {
                this.writeInteractionBuffer();
            }
            this.randomAccess.seek(4L);
            this.randomAccess.writeBoolean(true);
            this.close();
            this.readOnly = true;
            try {
                this.open();
            }
            catch (IOException ex) {
                throw new RuntimeException(ex);
            }
        }
    }

    public long getQtlZscoresSet() {
        return this.qtlZscoresSet;
    }

    public long getInteractionZscoresSet() {
        return this.interactionZscoresSet;
    }

    public long getQtlZscoresRead() {
        return this.qtlZscoresRead;
    }

    public long getInteractionZscoresRead() {
        return this.interactionZscoresRead;
    }

    public long getInteractionWriteBufferFlushed() {
        return this.interactionWriteBufferFlushed;
    }

    public long getQtlWriteBufferFlushed() {
        return this.qtlWriteBufferFlushed;
    }

    public long getInteractionReadBufferLoaded() {
        return this.interactionReadBufferLoaded;
    }

    public long getQtlReadBufferLoaded() {
        return this.qtlReadBufferLoaded;
    }

    @Override
    public void close() throws IOException {
        this.channel.close();
        this.randomAccess.close();
    }

    private void open() throws FileNotFoundException, IOException {
        this.randomAccess = new RandomAccessFile(this.interactionFile, this.readOnly ? "r" : "rw");
        if (!this.readOnly) {
            this.randomAccess.seek(4L);
            this.randomAccess.writeBoolean(false);
        }
        this.channel = this.randomAccess.getChannel();
    }

    private long getQtlPointer(String variantName, String geneName) throws BinaryInteractionFileException {
        int variantIndex = this.variantMap.get((Object)variantName);
        int geneIndex = this.genesMap.get((Object)geneName);
        if (variantIndex < 0) {
            throw new BinaryInteractionFileException("Variant not found: " + variantName);
        }
        if (geneIndex < 0) {
            throw new BinaryInteractionFileException("Gene not found: " + geneName);
        }
        int geneIndexInVariant = this.variants[variantIndex].getIndexOfGenePointer(geneIndex);
        if (geneIndexInVariant < 0) {
            throw new BinaryInteractionFileException("Cannot find QTL for: " + variantName + "-" + geneName);
        }
        return this.startQtlBlock + (long)(this.cummulativeGeneCountUpToVariant[variantIndex] + geneIndexInVariant) * this.sizeQtlBlock;
    }

    private long getInteractionPointer(String variantName, String geneName, String covariateName) throws BinaryInteractionFileException {
        int variantGeneCovariateIndex;
        int variantIndex = this.variantMap.get((Object)variantName);
        int geneIndex = this.genesMap.get((Object)geneName);
        int covariateIndex = this.covariatesMap.get((Object)covariateName);
        if (variantIndex < 0) {
            throw new BinaryInteractionFileException("Variant not found: " + variantName);
        }
        if (geneIndex < 0) {
            throw new BinaryInteractionFileException("Gene not found: " + geneName);
        }
        if (covariateIndex < 0) {
            throw new BinaryInteractionFileException("Covariate not found: " + covariateName);
        }
        int geneIndexInVariant = this.variants[variantIndex].getIndexOfGenePointer(geneIndex);
        if (geneIndexInVariant < 0) {
            throw new BinaryInteractionFileException("Cannot find variant gene combination for: " + variantName + "-" + geneName);
        }
        int variantGeneIndex = this.cummulativeGeneCountUpToVariant[variantIndex] + geneIndexInVariant;
        if (this.allCovariants) {
            variantGeneCovariateIndex = covariateIndex;
        } else {
            variantGeneCovariateIndex = Arrays.binarySearch(this.covariatesTested[variantGeneIndex], covariateIndex);
            if (variantGeneCovariateIndex < 0) {
                throw new BinaryInteractionFileException("Cannot find covariate " + covariateName + " for: " + variantName + "-" + geneName);
            }
        }
        return this.startInteractionBlock + (this.cummalitiveInteractionCountUptoVariantGene[variantGeneIndex] + (long)variantGeneCovariateIndex) * this.sizeInteractionBlock;
    }

    public BinaryInteractionQtlZscores readQtlResults(String variantName, String geneName) throws BinaryInteractionFileException, IOException {
        if (!this.normalQtlStored) {
            throw new BinaryInteractionFileException("This file does not store normal QTL restuls");
        }
        long qtlPointer = this.getQtlPointer(variantName, geneName);
        this.setQtlBuffer(qtlPointer, false);
        double[] zscore = new double[this.cohorts.length];
        for (int i = 0; i < this.cohorts.length; ++i) {
            zscore[i] = this.qtlBuffer.getDouble();
        }
        int[] sampleCounts = new int[this.cohorts.length];
        for (int i = 0; i < this.cohorts.length; ++i) {
            sampleCounts[i] = this.qtlBuffer.getInt();
        }
        ++this.qtlZscoresRead;
        if (this.metaAnalysis) {
            double metaZscore = this.qtlBuffer.getDouble();
            return new BinaryInteractionQtlZscores(zscore, sampleCounts, metaZscore);
        }
        return new BinaryInteractionQtlZscores(zscore, sampleCounts);
    }

    public void setQtlResults(String variantName, String geneName, BinaryInteractionQtlZscores zscores) throws BinaryInteractionFileException, IOException {
        int i;
        if (!this.normalQtlStored) {
            throw new BinaryInteractionFileException("This file does not store normal QTL restuls");
        }
        if (zscores.getZscores().length != this.cohorts.length) {
            throw new BinaryInteractionFileException("Error setting qtl " + variantName + "-" + geneName + " expected " + this.cohorts.length + " but found " + zscores.getZscores().length + " Z-scores");
        }
        if (zscores.getSampleCounts().length != this.cohorts.length) {
            throw new BinaryInteractionFileException("Error setting qtl " + variantName + "-" + geneName + " expected " + this.cohorts.length + " but found " + zscores.getSampleCounts().length + " samples counts");
        }
        long qtlPointer = this.getQtlPointer(variantName, geneName);
        this.setQtlBuffer(qtlPointer, true);
        for (i = 0; i < this.cohorts.length; ++i) {
            this.qtlBuffer.putDouble(zscores.getZscores()[i]);
        }
        for (i = 0; i < this.cohorts.length; ++i) {
            this.qtlBuffer.putInt(zscores.getSampleCounts()[i]);
        }
        if (this.metaAnalysis) {
            this.qtlBuffer.putDouble(zscores.getMetaZscore());
        }
        ++this.qtlZscoresSet;
    }

    private void setQtlBuffer(long qtlPointer, boolean writing) throws BinaryInteractionFileException, IOException {
        if (writing) {
            if (!this.qtlBufferWriting || qtlPointer != this.qtlBufferStart + (long)this.qtlBuffer.position() || (long)this.qtlBuffer.remaining() < this.sizeQtlBlock) {
                if (this.qtlBufferWriting) {
                    this.writeQtlBuffer();
                }
                this.qtlBuffer.clear();
                this.qtlBufferWriting = true;
                this.qtlBufferStart = qtlPointer;
            }
        } else {
            if (this.qtlBufferWriting) {
                this.writeQtlBuffer();
            }
            if (qtlPointer >= this.qtlBufferStart && qtlPointer + this.sizeQtlBlock <= this.qtlBufferStart + (long)this.qtlBuffer.limit()) {
                int positionInBuffer = (int)(qtlPointer - this.qtlBufferStart);
                this.qtlBuffer.position(positionInBuffer);
            } else {
                this.channel.position(qtlPointer);
                this.qtlBuffer.clear();
                this.channel.read(this.qtlBuffer);
                this.qtlBuffer.flip();
                this.qtlBufferStart = qtlPointer;
                ++this.qtlReadBufferLoaded;
            }
        }
    }

    private void writeQtlBuffer() throws BinaryInteractionFileException, IOException {
        if (this.readOnly) {
            throw new BinaryInteractionFileException("Interaction file is in read only mode");
        }
        if (!this.qtlBufferWriting) {
            throw new BinaryInteractionFileException("Interaction file has no QTLs to write");
        }
        this.channel.position(this.qtlBufferStart);
        this.qtlBuffer.flip();
        this.channel.write(this.qtlBuffer);
        this.qtlBufferWriting = false;
        ++this.qtlWriteBufferFlushed;
    }

    public BinaryInteractionZscores readInteractionResults(String variantName, String geneName, String covariateName) throws BinaryInteractionFileException, IOException {
        long interactionPointer = this.getInteractionPointer(variantName, geneName, covariateName);
        return this.readInteractionResults(interactionPointer);
    }

    public BinaryInteractionQueryResult readVariantGeneCovariateResults(String variantName, String geneName, String covariateName) throws BinaryInteractionFileException, IOException {
        BinaryInteractionQtlZscores qtlZscores = this.isNormalQtlStored() ? this.readQtlResults(variantName, geneName) : null;
        BinaryInteractionZscores interactionRestuls = this.readInteractionResults(variantName, geneName, covariateName);
        return new BinaryInteractionQueryResult(variantName, geneName, covariateName, qtlZscores, interactionRestuls);
    }

    public Iterator<BinaryInteractionQueryResult> readVariantGeneResults(String variantName, String geneName) throws BinaryInteractionFileException, IOException {
        int variantIndex = this.variantMap.get((Object)variantName);
        int geneIndex = this.genesMap.get((Object)geneName);
        if (variantIndex < 0) {
            throw new BinaryInteractionFileException("Variant not found: " + variantName);
        }
        if (geneIndex < 0) {
            throw new BinaryInteractionFileException("Gene not found: " + geneName);
        }
        int geneIndexInVariant = this.variants[variantIndex].getIndexOfGenePointer(geneIndex);
        if (geneIndexInVariant < 0) {
            throw new BinaryInteractionFileException("Cannot find variant gene combination for: " + variantName + "-" + geneName);
        }
        int variantGeneIndex = this.cummulativeGeneCountUpToVariant[variantIndex] + geneIndexInVariant;
        BinaryInteractionQtlZscores qtlZscore = this.isNormalQtlStored() ? this.readQtlResults(variantName, geneName) : null;
        long startVariantGeneBlock = this.startInteractionBlock + this.cummalitiveInteractionCountUptoVariantGene[variantGeneIndex] * this.sizeInteractionBlock;
        return new BinaryInteractionCovariateIterator(variantName, geneName, this.allCovariants ? null : this.covariatesTested[variantGeneIndex], this, startVariantGeneBlock, qtlZscore);
    }

    protected BinaryInteractionZscores readInteractionResults(long interactionPointer) throws BinaryInteractionFileException, IOException {
        double[] zscoreInteractionFlippedCohort;
        this.setInteactionBuffer(interactionPointer, false);
        ++this.interactionZscoresRead;
        int[] samplesInteractionCohort = this.readIntArrayFromInteractionBuffer(this.cohorts.length);
        double[] zscoreSnpCohort = this.readDoubleArrayFromInteractionBuffer(this.cohorts.length);
        double[] zscoreCovariateCohort = this.readDoubleArrayFromInteractionBuffer(this.cohorts.length);
        double[] zscoreInteractionCohort = this.readDoubleArrayFromInteractionBuffer(this.cohorts.length);
        double[] rSquaredCohort = this.readDoubleArrayFromInteractionBuffer(this.cohorts.length);
        if (this.flippedZscoreStored) {
            zscoreInteractionFlippedCohort = this.readDoubleArrayFromInteractionBuffer(this.cohorts.length);
        } else {
            zscoreInteractionFlippedCohort = new double[this.cohorts.length];
            Arrays.fill(zscoreInteractionFlippedCohort, Double.NaN);
        }
        if (this.metaAnalysis) {
            double zscoreSnpMeta = this.interactionBuffer.getDouble();
            double zscoreCovariateMeta = this.interactionBuffer.getDouble();
            double zscoreInteractionMeta = this.interactionBuffer.getDouble();
            if (this.flippedZscoreStored) {
                double zscoreInteractionFlippedMeta = this.interactionBuffer.getDouble();
                return new BinaryInteractionZscores(samplesInteractionCohort, zscoreSnpCohort, zscoreCovariateCohort, zscoreInteractionCohort, rSquaredCohort, zscoreInteractionFlippedCohort, zscoreSnpMeta, zscoreCovariateMeta, zscoreInteractionMeta, zscoreInteractionFlippedMeta);
            }
            return new BinaryInteractionZscores(samplesInteractionCohort, zscoreSnpCohort, zscoreCovariateCohort, zscoreInteractionCohort, rSquaredCohort, zscoreSnpMeta, zscoreCovariateMeta, zscoreInteractionMeta);
        }
        return new BinaryInteractionZscores(samplesInteractionCohort, zscoreSnpCohort, zscoreCovariateCohort, zscoreInteractionCohort, rSquaredCohort, zscoreInteractionFlippedCohort);
    }

    public void setInteractionResults(String variantName, String geneName, String covariateName, BinaryInteractionZscores zscores) throws BinaryInteractionFileException, IOException {
        if (zscores.getSamplesInteractionCohort().length != this.cohorts.length) {
            throw new BinaryInteractionFileException("Error setting interaction " + variantName + "-" + geneName + " expected " + this.cohorts.length + " but found " + zscores.getSamplesInteractionCohort().length + " cohorts");
        }
        long interactionPointer = this.getInteractionPointer(variantName, geneName, covariateName);
        this.setInteactionBuffer(interactionPointer, true);
        this.writeIntArrayToInteractionBuffer(zscores.getSamplesInteractionCohort());
        this.writeDoubleArrayToInteractionBuffer(zscores.getZscoreSnpCohort());
        this.writeDoubleArrayToInteractionBuffer(zscores.getZscoreCovariateCohort());
        this.writeDoubleArrayToInteractionBuffer(zscores.getZscoreInteractionCohort());
        this.writeDoubleArrayToInteractionBuffer(zscores.getrSquaredCohort());
        if (this.flippedZscoreStored) {
            this.writeDoubleArrayToInteractionBuffer(zscores.getZscoreInteractionFlippedCohort());
        }
        if (this.metaAnalysis) {
            this.interactionBuffer.putDouble(zscores.getZscoreSnpMeta());
            this.interactionBuffer.putDouble(zscores.getZscoreCovariateMeta());
            this.interactionBuffer.putDouble(zscores.getZscoreInteractionMeta());
            if (this.flippedZscoreStored) {
                this.interactionBuffer.putDouble(zscores.getZscoreInteractionFlippedMeta());
            }
        }
        ++this.interactionZscoresSet;
    }

    private void setInteactionBuffer(long interactionPointer, boolean writing) throws BinaryInteractionFileException, IOException {
        if (writing) {
            if (!this.interactionBufferWriting || interactionPointer != this.interactionBufferStart + (long)this.interactionBuffer.position() || (long)this.interactionBuffer.remaining() < this.sizeInteractionBlock) {
                if (this.interactionBufferWriting) {
                    this.writeInteractionBuffer();
                }
                this.interactionBuffer.clear();
                this.interactionBufferWriting = true;
                this.interactionBufferStart = interactionPointer;
            }
        } else {
            if (this.interactionBufferWriting) {
                this.writeInteractionBuffer();
            }
            if (interactionPointer >= this.interactionBufferStart && interactionPointer + this.sizeInteractionBlock <= this.interactionBufferStart + (long)this.interactionBuffer.limit()) {
                int positionInBuffer = (int)(interactionPointer - this.interactionBufferStart);
                this.interactionBuffer.position(positionInBuffer);
            } else {
                this.channel.position(interactionPointer);
                this.interactionBuffer.clear();
                this.channel.read(this.interactionBuffer);
                this.interactionBuffer.flip();
                this.interactionBufferStart = interactionPointer;
                ++this.interactionReadBufferLoaded;
            }
        }
    }

    private void writeInteractionBuffer() throws BinaryInteractionFileException, IOException {
        if (this.readOnly) {
            throw new BinaryInteractionFileException("Interaction file is in read only mode");
        }
        if (!this.interactionBufferWriting) {
            throw new BinaryInteractionFileException("Interaction file has no interactions to write");
        }
        this.channel.position(this.interactionBufferStart);
        this.interactionBuffer.flip();
        this.channel.write(this.interactionBuffer);
        this.interactionBufferWriting = false;
        ++this.interactionWriteBufferFlushed;
    }

    private double[] readDoubleArrayFromInteractionBuffer(int length) {
        if (length == 0) {
            return BinaryInteractionZscores.emptyDoubleArray;
        }
        double[] array = new double[length];
        for (int i = 0; i < length; ++i) {
            array[i] = this.interactionBuffer.getDouble();
        }
        return array;
    }

    private int[] readIntArrayFromInteractionBuffer(int length) {
        if (length == 0) {
            return BinaryInteractionZscores.emptyIntArray;
        }
        int[] array = new int[length];
        for (int i = 0; i < length; ++i) {
            array[i] = this.interactionBuffer.getInt();
        }
        return array;
    }

    private void writeDoubleArrayToInteractionBuffer(double[] array) {
        for (int i = 0; i < array.length; ++i) {
            this.interactionBuffer.putDouble(array[i]);
        }
    }

    private void writeIntArrayToInteractionBuffer(int[] array) {
        for (int i = 0; i < array.length; ++i) {
            this.interactionBuffer.putInt(array[i]);
        }
    }

    public int getGeneCount() {
        return this.genes.length;
    }

    public int getCovariateCount() {
        return this.covariates.length;
    }

    public int getVariantCount() {
        return this.variants.length;
    }

    public int getCohortCount() {
        return this.cohorts.length;
    }

    public int getVariantGeneCombinations() {
        return this.cummalitiveInteractionCountUptoVariantGene.length - 1;
    }

    public boolean containsGene(String geneName) {
        return this.genesMap.containsKey((Object)geneName);
    }

    public boolean containsVariant(String variantName) {
        return this.variantMap.containsKey((Object)variantName);
    }

    public boolean containsCovariant(String covariateName) {
        return this.covariatesMap.containsKey((Object)covariateName);
    }

    public boolean containsVariantGene(String variantName, String geneName) {
        int variantIndex = this.variantMap.get((Object)variantName);
        int geneIndex = this.genesMap.get((Object)geneName);
        if (variantIndex < 0) {
            return false;
        }
        if (geneIndex < 0) {
            return false;
        }
        int geneIndexInVariant = this.variants[variantIndex].getIndexOfGenePointer(geneIndex);
        return geneIndexInVariant >= 0;
    }

    public boolean containsInteraction(String variantName, String geneName, String covariateName) throws BinaryInteractionFileException {
        int variantIndex = this.variantMap.get((Object)variantName);
        int geneIndex = this.genesMap.get((Object)geneName);
        int covariateIndex = this.covariatesMap.get((Object)covariateName);
        if (variantIndex < 0) {
            return false;
        }
        if (geneIndex < 0) {
            return false;
        }
        if (covariateIndex < 0) {
            return false;
        }
        int geneIndexInVariant = this.variants[variantIndex].getIndexOfGenePointer(geneIndex);
        if (geneIndexInVariant < 0) {
            return false;
        }
        if (this.allCovariants) {
            return true;
        }
        int variantGeneIndex = this.cummulativeGeneCountUpToVariant[variantIndex] + geneIndexInVariant;
        int variantGeneCovariateIndex = Arrays.binarySearch(this.covariatesTested[variantGeneIndex], covariateIndex);
        return variantGeneCovariateIndex >= 0;
    }

    private void createVariantChrPosMap() {
        this.variantsByPos = new ChrPosMap();
        for (BinaryInteractionVariant variant : this.variants) {
            this.variantsByPos.put(variant.getChr(), variant.getPos(), variant);
        }
    }

    public BinaryInteractionVariant getVariant(String chr, int pos) {
        if (this.variantsByPos == null) {
            this.createVariantChrPosMap();
        }
        return this.variantsByPos.get(chr, pos);
    }
}

